#! /usr/bin/env python3
#
# Copyright (C) 2023 MicroSys Electronics GmbH
# Author: Kay Potthoff <kay.potthoff@microsys.de>
#

# This tool migrates NXP repository URLs that had been hosted
# on Codeaurora to their new locations.
# Together with this tool you should have received a copy of
# README.md which contains detailed instruction how to use this
# tool.
# You have read and understood README.md before using this tool.

import sys
import getopt
import os
import subprocess
import glob
#import re
from contextlib import contextmanager

__version__ = "1.0.4"

repo_tool = "repo"

# List of URLs and their replacements
# Note that the order in this list matters
replace_urls = {
        "source.codeaurora.org/external/qoriq/qoriq-components":
            "github.com/nxp-qoriq",

        "source.codeaurora.org/external/imx":
            "github.com/nxp-imx",

        "source.codeaurora.org/external/qoriq/qoriq-yocto-sdk":
            "github.com/nxp-qoriq-yocto-sdk",

        "source.codeaurora.org/external/autoivnsw/sja1110_linux":
            "github.com/nxp-archive/autoivnsw_sja1110_linux",

        "source.codeaurora.org/external/autobsps32/extra/pfeng":
            "github.com/nxp-auto-linux/pfeng",

        "source.codeaurora.org/external/autobsps32/ipcf/ipc-shm":
            "github.com/nxp-auto-linux/ipc-shm",

        "source.codeaurora.org/external/autobsps32/extra/sm_drv":
            "github.com/nxp-archive/autobsps32_sm_drv",

        "source.codeaurora.org/external/autobsps32":
            "github.com/nxp-auto-linux",

        "source.codeaurora.org/external/qoriq":
            "github.com/nxp-qoriq",

        "source.codeaurora.org/external/qoriq/qoriq-components/meta-qoriq":
            "github.com/nxp-qoriq/meta-qoriq",

        "source.codeaurora.org/external/autobsps32/meta-alb":
            "github.com/nxp-auto-linux/meta-alb",

        "source.codeaurora.org/external/s32ds/compiler/gnu_nxp/plain":
            "raw.githubusercontent.com/nxp-auto-tools/gnu_nxp/master",

        "source.codeaurora.org/external/autobsps32/extra/sm_drv":
            "github.com/nxp-archive/autobsps32_sm_drv",

        # Some special handling is necessary for meta-qoriq:

        "qoriq-components/meta-qoriq": "meta-qoriq",

        "source.codeaurora.org/external":
            "github.com/nxp-qoriq",

        #"qoriq/meta-qoriq": "meta-qoriq",
    }

# List of URLs where the branch 'master' has been
# renamed to 'main'
git_branches = {
    "git://github.com/opencontainers/runc": ("master", "main"),
    "git://github.com/cracklib/cracklib":   ("master", "main"),
    "git://github.com/gorilla/context":     ("master", "main"),
    "git://github.com/gorilla/mux":         ("master", "main"),
    "git://github.com/coreos/go-systemd":   ("master", "main"),
}

def usage():
    prog = os.path.basename(sys.argv[0])
    print("{0} {1}\n".format(prog, __version__))
    print("usage: {0} [options] ...".format(prog))
    print("\nOptions:")
    print("  -h|--help                  print this help")
    print("  -u|--manifest-url <url>          use <url> for 'repo'")
    print("  -b|--manifest-branch <revision>  use <revision> as branch for 'repo'")
    print("  -m|--manifest-name <name>        use <name> for repo's manifest")
    print("  --sdk-path <path>                use <path> for SDK path")
    print("  --manifest-path <path>           use <path> for repo's manifest path")
    print("  --repo <pathname>                use alternative repo command")
    print("  --fix-path <path>                fixes URLs in <path>")

@contextmanager
def goto_dir(path):
    cwd = os.getcwd()
    try:
        os.chdir(path)
        yield
    finally:
        os.chdir(cwd)

def do_replace_url(s):
    if (not s) or (len(s) == 0):
        return s
    tmp_s = s
    for url in replace_urls:
        if url in tmp_s:
            tmp_s = tmp_s.replace(url, replace_urls[url])
    return tmp_s

def git(cmd):
    c = ["git"]
    c.extend(cmd)
    return subprocess.run(c).returncode

def repo(cmd):
    c = [repo_tool]
    c.extend(cmd)
    return subprocess.run(c).returncode

def fix_urls(path):

    print("Fixing URLs in '{p}' ...".format(p=path))

    extensions = [".xml", ".bb*", ".inc", ".conf"]

    with goto_dir(path):
        flist = []
        for suffix in extensions:
            flist.extend([x for x in glob.glob("**/*"+suffix, recursive=True)])

        for f in flist:
            if os.path.islink(f):
                continue
            ln = []
            with open(f, "r") as fh:
                ln = fh.readlines()
            fixed = False
            for i in range(len(ln)):
                for u in replace_urls:
                    if u in ln[i]:
                        ln[i] = ln[i].replace(u, replace_urls[u])
                        fixed = True
            if fixed:
                with open(f, "w") as fh:
                    for l in ln:
                        print(l, end='', file=fh)
                os.chmod(f, 0o644)
                print("Fixed {0}".format(f))

def fix_branches(path):

    print("Fixing branches in '{p}' ...".format(p=path))

    extensions = [".bb*", ".inc"]

    with goto_dir(path):
        flist = []
        for suffix in extensions:
            flist.extend([x for x in glob.glob("**/*"+suffix, recursive=True)])

        for f in flist:
            if os.path.islink(f):
                continue
            ln = []
            with open(f, "r") as fh:
                ln = fh.readlines()
            fixed = False

            for i in range(len(ln)):
                for u in git_branches:
                    if u in ln[i]:
                        br_master = git_branches[u][0]
                        if ("branch="+br_master) in ln[i]:
                            br_main   = git_branches[u][1]
                            ln[i] = ln[i].replace("branch="+br_master,
                                                  "branch="+br_main)
                            fixed = True

            if fixed:
                with open(f, "w") as fh:
                    for l in ln:
                        print(l, end='', file=fh)
                os.chmod(f, 0o644)
                print("Fixed {0}".format(f))

def do_fix_path(fix_path):
    fix_urls(fix_path)
    fix_branches(fix_path)
    return 0

def main():

    sdk_path = "." # default is the current directory

    fix_path = "."

    manifest = {
        "name": "default.xml",
        "branch": "HEAD",
        "path": os.path.join(sdk_path, ".manifest")
    }

    try:
        opts, argv = getopt.getopt(sys.argv[1:],
                                   "h,u:,b:,m:",
                                   ["help",
                                    "manifest-url=",
                                    "manifest-branch=",
                                    "manifest-name=",
                                    "sdk-path=",
                                    "manifest-path=",
                                    "repo=",
                                    "fix-path="])
    except getopt.GetoptError as err:
        print("Error: {0}".format(err), file=sys.stderr)
        usage()
        sys.exit(1)

    for option, optarg in opts:
        if option in ("-h", "--help"):
            usage()
            sys.exit(0)
        elif option in ("-u", "--manifest-url"):
            manifest["url"] = optarg
        elif option in ("-b", "--manifest-branch"):
            manifest["branch"] = optarg
        elif option in ("-m", "--manifest-name"):
            manifest["name"] = optarg
        elif option in ("--sdk-path"):
            sdk_path = optarg
        elif option in ("--manifest-path"):
            manifest["path"] = optarg
        elif option in ("--repo"):
            repo_tool = optarg
        elif option in ("--fix-path"):
            fix_path = optarg
            return do_fix_path(fix_path)
        else:
            print("Error: unknown option \"{0}\"!".format(option),
                  file=sys.stderr)
            sys.exit(1)

    if not "url" in manifest:
        print("Error: no manifest URL given!", file=sys.stderr)
        return 1

    if not "branch" in manifest:
        print("Error: no release or branch specified!", file=sys.stderr)
        return 1

    manifest["url"] = do_replace_url(manifest["url"])

    print("Repository: {url}".format(url=manifest["url"]))

    if sdk_path != ".":

        if os.path.isdir(sdk_path):
            print("Error: directory '{sdk}' already exists!"
                  .format(sdk=sdk_path),
                  file=sys.stderr)
            print("Please delete the directory or choose a different one.")
            return 1

    if os.path.isdir(manifest["path"]):
        print("Error: directory '{manifest}' already exists!"
              .format(manifest=manifest["path"]),
              file=sys.stderr)
        print("Please delete the directory or choose a different one.")
        return 1

    rc = git(["clone", manifest["url"], manifest["path"]])
    if rc != 0:
        print("Error: cannot clone repository!", file=sys.stderr)
        return rc

    with goto_dir(manifest["path"]):
        rc = git(["checkout", manifest["branch"]])
        if rc != 0:
            print("Error: cannot checkout {release}!"
                  .format(release=manifest["branch"]),
                  file=sys.stderr)
            return rc

    fix_urls(manifest["path"])

    # re-direct URL to point to the directory:
    manifest["url"] = "file://"+os.path.abspath(manifest["path"])
    manifest["name"] = os.path.join(os.path.abspath(manifest["path"]),
                                    manifest["name"])

    if not os.path.isdir(sdk_path):
        os.makedirs(sdk_path)

    with goto_dir(sdk_path):
        rc = repo(["init",
                   "--no-clone-bundle",
                   "--manifest-url="+manifest["url"],
                   "--manifest-branch="+manifest["branch"],
                   "--manifest-name="+manifest["name"]])
        if rc != 0:
            print("Error: cannot init {url}!".format(url=manifest["url"]),
                  file=sys.stderr)
            return rc

        rc = repo(["sync",
                   "--current-branch",
                   "--jobs=4",
                   "--fail-fast",
                   "--no-clone-bundle",
                   "--force-sync",
                   "--no-manifest-update",
                   "--auto-gc",
                   "--manifest-name="+manifest["name"]])
        if rc != 0:
            print("Error: cannot sync {url}!".format(url=manifest["url"]),
                  file=sys.stderr)
            return rc

    fix_urls(sdk_path)
    fix_branches(sdk_path)

    print("========================================")
    print("The URLs have been fixed in all recipes!")

    if sdk_path != ".":
        print("You can now go to the directory {sdk} and".format(sdk=sdk_path))
        print("continue with your build tasks.")
    else:
        print("You can now continue with your build tasks.")

    return 0

if __name__ == "__main__":
    sys.exit(main())

#######################################
# Local Variables:
# mode: Python
# tab-width: 4
# indent-tabs-mode: nil
# End:
# kate: space-indent on; indent-width 4; mixedindent off; indent-mode python;
# vi: set et tabstop=4 shiftwidth=4:
# vim: set expandtab filetype=python:
