#! /usr/bin/python3
#
# Copyright (C) 2024-2025 MicroSys Electronics GmbH
# Author: Kay Potthoff <kay.potthoff@microsys.de>
#

import sys
import getopt
import os
import glob
import subprocess
from pathlib import Path

try:
    import libfdt
    has_libfdt = True
except ModuleNotFoundError:
    has_libfdt = False

__version__ = "1.0.1"

HASH_ALGO = "sha256"

target_arch = {
    "powerpc64": "powerpc",
    "aarch64":   "arm64"
}

class DtsEntry:

    def __init__(self, name, indent=0):
        self.name = name
        self.indent = int(indent)
        self.indent_step = 4
        self.indent_stack = []
        self.parent = None

    def getName(self):
        return self.name

    def setName(self, name):
        self.name = name

    def getParent(self):
        return self.parent

    def setParent(self, parent):
        self.parent = parent

    def pushIndent(self, next_indent=None):
        self.indent_stack.append(self.getIndent())
        if next_indent:
            self.setIndent(next_indent)

    def popIndent(self):
        if len(self.indent_stack) > 0:
            self.setIndent(self.indent_stack.pop())

    def setIndent(self, indent):
        if indent >= 0:
            self.indent = int(indent)
        else:
            self.indent = 0

    def getIndent(self):
        return self.indent

    def resetIndent(self):
        self.setIndent(0)

    def incIndent(self, n=1):
        self.indent += n

    def decIndent(self, n=1):
        self.indent -= n

    def tab(self):
        return " " * (self.indent*self.indent_step)

    def write(self, fh):
        raise NotImplementedError()

    def isNode(self):
        return False

class DtsProperty(DtsEntry):

    def __init__(self, name, value=None):
        super().__init__(name)
        self.value = value

    def __str__(self):
        return str(self.getValue())

    def getValue(self):
        return self.value

    def setValue(self, value):
        self.value = value

    def emit(self):
        return self.tab() + self.getName() + " = " + str(self)

    def write(self, fh):
        print(self.emit()+";", file=fh)

class DtsStringProperty(DtsProperty):

    def __init__(self, name, value=None):
        if not value:
            value = ''
        super().__init__(name, value)

    def __str__(self):
        return '"' + str(self.getValue()) + '"'

class DtsBooleanProperty(DtsProperty):

    def __init__(self, name):
        super().__init__(name, True)

    def emit(self):
        return self.tab() + self.getName()

class DtsIncBinProperty(DtsProperty):

    def __init__(self, name, value=None):
        if not value:
            value = ''
        super().__init__(name, value)

    def __str__(self):
        return '/incbin/("' + str(self.getValue()) + '")'

class DtsIntBaseProperty(DtsProperty):

    def __init__(self, name, value=None, base=0):
        super().__init__(name, value)
        self.base = base
        self.cells = -1

    def getBase(self):
        return self.base

    def setBase(self, base):
        self.base = base

    def getCells(self):
        return self.cells

    def setCells(self, cells):
        self.cells = cells

    def getFormatStr(self):
        if (self.base == 0) or (self.base == 10):
            return "{0:d}"
        elif (self.base == 16):
            return "0x{0:x}"
        else:
            return "{0}"

class DtsIntProperty(DtsIntBaseProperty):

    def __init__(self, name, value=None, base=0):
        if value is None:
            value = -1
        if isinstance(value, str):
            value = eval(value)
        super().__init__(name, value, base)
        if (name == "offset") or (name == "reg") \
            or (name == "entry") or (name == "load") \
            or ("size" in name):
            self.setBase(16)
        else:
            self.setBase(10)

    def setValue(self, value):
        if value is None:
            value = -1
        if isinstance(value, str):
            value = eval(value)
        super().setValue(value)
        if "size" in self.getName() and self.getValue() > 0xffffffff:
            parent = self.getParent()
            if parent and hasattr(parent, 'getSizeCells'):
                cells = parent.getSizeCells()
                if cells > 1:
                    self.setCells(cells)

    def __str__(self):
        fmt = self.getFormatStr()
        if self.getCells() > 1:
            prop = []
            for shift in reversed(range(self.getCells())):
                val = fmt.format((self.getValue() >> (shift*32))&0xffffffff)
                prop.append(val)
            return "<"  + " ".join(prop) +  ">"
        return "<" + fmt.format(self.getValue()) + ">"

class DtsAddressProperty(DtsIntProperty):

    def __init__(self, name, value=0, base=16):
        super().__init__(name, value, base)
        if self.getValue() > 0xffffffff:
            self.setCells(2)

    def setValue(self, value):
        super().setValue(value)
        if self.getValue() > 0xffffffff:
            self.setCells(2)

    def getFormatStr(self):
        return "0x{0:08x}"

class DtsIntListProperty(DtsIntBaseProperty):

    def __init__(self, name, value=None):
        if not value:
            value = []
        if not isinstance(value, list):
            value = list(value)
        vl = []
        for v in value:
            if isinstance(v, str):
                v = int(v, 0)
            vl.append(v)
        super().__init__(name, vl)
        self.setBase(16)

    def __str__(self):
        fmt = self.getFormatStr()
        a = [fmt.format(v) for v in self.getValue()]
        return "<" + " ".join(a) + ">"

class DtsRegProperty(DtsIntListProperty):

    def __init__(self, address, size):
        super().__init__("reg")
        self.setBase(16)
        self.setAddress(address)
        self.setSize(size)

    def setAddress(self, address):
        self.address = address

    def getAddress(self):
        return self.address

    def setSize(self, size):
        self.size = size

    def getSize(self):
        return self.size

    def getValue(self):
        adr_w = 1
        sz_w = 1
        parent = self.getParent()
        if parent:
            w = parent.getAddressCells()
            if w > 0:
                adr_w = w
            w = parent.getSizeCells()
            if w > 0:
                sz_w = w
        vl = []
        for shift in reversed(range(adr_w)):
            a = (self.address >> (shift*32)) & 0xffffffff
            vl.append(a)
        for shift in reversed(range(sz_w)):
            a = (self.size >> (shift*32)) & 0xffffffff
            vl.append(a)
        return vl

class DtsNode(DtsEntry):

    def __init__(self, name, address_cells=-1, size_cells=-1):
        super().__init__(name)
        self.child = {}
        self.setAddressCells(address_cells)
        self.setSizeCells(size_cells)

    def __str__(self):
        return str(self.child)

    def isNode(self):
        return True

    def setAddressCells(self, address_cells):
        self.address_cells = address_cells
        if address_cells > 0:
            self["#address-cells"] = address_cells

    def getAddressCells(self):
        if self.address_cells > 0:
            return self.address_cells
        parent = self.getParent()
        if parent:
            return parent.getAddressCells()
        return 1

    def setSizeCells(self, size_cells):
        self.size_cells = size_cells
        if size_cells > 0:
            self["#size-cells"] = size_cells

    def getSizeCells(self):
        if self.size_cells > 0:
            return self.size_cells
        parent = self.getParent()
        if parent:
            return parent.getSizeCells()
        return 1

    def set(self, entry):
        self.child[entry.getName()] = entry
        entry.setParent(self)

    def get(self, name):
        return self.child[name]

    def __contains__(self, item):
        return item in self.child

    def __getitem__(self, name):
        return self.get(name)

    def __setitem__(self, name, value):
        if isinstance(value, str):
            self.set(DtsStringProperty(name, value))
        elif isinstance(value, bool):
            self.set(DtsBooleanProperty(name))
        elif isinstance(value, int):
            self.set(DtsIntProperty(name, value))
        elif isinstance(value, list):
            self.set(DtsIntListProperty(name, value))
        elif isinstance(value, DtsProperty):
            self.set(value)
        elif isinstance(value, DtsEntry):
            if name in self.child:
                self.child[name].set(value)

    def incIndent(self, n=1):
        super().incIndent(n)
        for c in self.child:
            self.child[c].incIndent()

    def decIndent(self, n=1):
        super().decIndent(n)
        for c in self.child:
            self.child[c].decIndent()

    def getChildren(self):
        return self.child

    def adjustAddressCells(self, cells):
        for c in self.child:
            if isinstance(self.child[c], DtsAddressProperty):
                self.child[c].setCells(cells)

    def write(self, fh):
        print("", file=fh)
        tab = self.tab()
        print(tab+self.getName()+ " {", file=fh)

        # Note: Properties must precede subnodes

        # 1. all properties:
        for c in self.child:
            if isinstance(self.child[c], DtsProperty):
                self.child[c].incIndent()
                self.child[c].write(fh)
                self.child[c].decIndent()

        # 2. all subnodes:
        for c in self.child:
            if isinstance(self.child[c], DtsNode):
                self.child[c].incIndent()
                self.child[c].write(fh)
                self.child[c].decIndent()

        print(tab+"};", file=fh)

    def getSubnodes(self):
        nodes = []
        for c in self.child:
            if isinstance(self.child[c], DtsNode):
                nodes.append(self.child[c])
        return nodes

    def parseParameter(self, param):
        args = param.split(':')
        image = args[0]
        args = args[1:]
        arg = {}
        for a in args:
            p = a.split('=')
            if len(p) > 1:
                arg[p[0]] = p[1]
            else:
                arg[p[0]] = True
        return (image, arg)

class DtsFITNode(DtsNode):

    def __init__(self, name, address_cells=-1, size_cells=-1):
        super().__init__(name, address_cells, size_cells)

        for n in ["images", "configurations"]:
            self.set(DtsNode(n))

    def addImage(self, name, description=None, hash=HASH_ALGO, arch=None):
        self["images"] = DtsNode(name)
        if description:
            self["images"][name]["description"] = description

        if arch:
            if arch in target_arch:
                arch = target_arch[arch]
            self["images"][name]["arch"] = arch
        else:
            self["images"][name]["arch"] = "arm64"

        if hash:
            self["images"][name] = DtsNode("hash-1")
            self["images"][name]["hash-1"]["algo"] = hash

        # Note:
        #'compression' nodes for ramdisks are deprecated
        if not name.startswith("ramdisk"):
            self["images"][name]["compression"] = "none"

        if name.startswith("kernel"):
            self["images"][name]["type"] = "kernel"
            self["images"][name]["os"] = "linux"

        if name.startswith("fdt"):
            self["images"][name]["type"] = "flat_dt"

        if name.startswith("ramdisk"):
            self["images"][name]["type"] = "ramdisk"
            self["images"][name]["os"] = "linux"

        if name.startswith("efi"):
            self["images"][name]["type"] = "kernel_noload"
            self["images"][name]["os"] = "efi"
            self["images"][name]["compression"] = "none"

    def setImageEntry(self, name, address):
        entry = DtsAddressProperty("entry", address)
        self["images"][name].set(entry)
        address = entry.getValue()
        if address > 0xffffffff:
            entry.setCells(2)
            self.setAddressCells(2)
        if name.startswith("kernel") \
            or name.startswith("efi") \
            or name.startswith("ramdisk"):
            self.setImageLoad(name, address)

    def setImageLoad(self, name, address):
        load = DtsAddressProperty("load", address)
        address = load.getValue()
        if address > 0xffffffff:
            load.setCells(2)
            self.setAddressCells(2)
        self["images"][name].set(load)

    def setImage(self, name, image, hash=HASH_ALGO):
        if not name in self["images"]:
            self.addImage(name, hash=hash)
        (image, arg) = self.parseParameter(image)
        if os.path.isfile(image):
            img = os.path.realpath(image)
            self["images"][name].set(DtsIncBinProperty("data", img))
        else:
            self["images"][name]["data"] = image
        # Note:
        #'compression' nodes for ramdisks are deprecated
        if image.endswith(".gz") and (not name.startswith("ramdisk")):
            self["images"][name]["compression"] = "gzip"
        if "entry" in arg:
            self.setImageEntry(name, arg["entry"])
        if "load" in arg:
            self.setImageLoad(name, arg["load"])
            if (not "entry" in arg) and name.startswith("ramdisk"):
                self["images"][name].set(DtsAddressProperty("entry", arg["load"]))

    def addConfiguration(self, name, description=None, hash=HASH_ALGO):
        self["configurations"] = DtsNode(name)
        if description:
            self["configurations"][name]["description"] = description
        else:
            self["configurations"][name]["description"] = f"configuration {name}"

        if hash:
            self["configurations"][name] = DtsNode("hash-1")
            self["configurations"][name]["hash-1"]["algo"] = hash

    def setDefaultConfiguration(self, config):
        self["configurations"]["default"] = config

    def finalizeConfigs(self, hash=HASH_ALGO):
        address_cells = self.getAddressCells()
        subnode = self["images"].getSubnodes()
        kernel = []
        fdt = []
        ramdisk = []
        for sn in subnode:
            sn.adjustAddressCells(address_cells)
            if sn.getName().startswith("kernel") \
                or sn.getName().startswith("efi"):
                kernel.append(sn)
            elif sn.getName().startswith("fdt"):
                fdt.append(sn)
            elif sn.getName().startswith("ramdisk"):
                ramdisk.append(sn)
        cfg_no = 0
        if len(ramdisk) > 0:
            for k in kernel:
                for f in fdt:
                    for r in ramdisk:
                        cfg = "ramboot-{0}".format(cfg_no)
                        self.addConfiguration(cfg, hash=hash)
                        if cfg_no == 0:
                            self.setDefaultConfiguration(cfg)
                        self["configurations"][cfg]["kernel"] = k.getName()
                        self["configurations"][cfg]["fdt"] = f.getName()
                        self["configurations"][cfg]["ramdisk"] = r.getName()
                        cfg_no = cfg_no + 1
        else:
            for k in kernel:
                for f in fdt:
                    compatible = None
                    image = Path(f["data"].getValue())
                    if image.is_file():
                        cfg = image.stem
                        dtb = DeviceTree(image)
                        compatible = dtb.getCompatible()
                    else:
                        cfg = "config-{0}".format(cfg_no)
                    self.addConfiguration(cfg, hash=hash)
                    if cfg_no == 0:
                        self.setDefaultConfiguration(cfg)
                    self["configurations"][cfg]["kernel"] = k.getName()
                    self["configurations"][cfg]["fdt"] = f.getName()
                    if compatible:
                        # TODO: set CONFIG_FIT_BEST_MATCH in U-Boot
                        comp = []
                        for c in compatible:
                            if ("microsys" in c) or ("mpx" in c):
                                comp.append(c)
                        if len(comp) > 0:
                            self["configurations"][cfg]["compatible"] = comp[0]
                    cfg_no = cfg_no + 1

class ITS:

    VERSION = "/dts-v1/"

    def __init__(self, root_node, description=None):
        self.setRoot(root_node)
        if description:
            self.root["description"] = description

    def setDescription(self, description):
        self.root["description"] = description

    def setRoot(self, root_node):
        self.root = root_node

    def getRoot(self):
        return self.root

    def setAddressCells(self, address_cells):
        self.getRoot().setAddressCells(address_cells)

    def getAddressCells(self):
        return self.getRoot().getAddressCells()

    def setSizeCells(self, size_cells):
        self.getRoot().setSizeCells(size_cells)

    def getSizeCells(self):
        return self.getRoot().getSizeCells()

    def set(self, entry):
        self.getRoot().set(entry)

    def get(self, name):
        return self.getRoot().get(name)

    def __getitem__(self, name):
        return self.get(name)

    def __setitem__(self, name, value):
        self.getRoot()[name] = value

    def write(self, fh):
        print("/* generated file; do not edit */", file=fh)
        print(self.VERSION+";", file=fh)
        self.getRoot().write(fh)

class FITImage(ITS):

    def __init__(self, description=None, address_cells=-1):
        super().__init__(DtsFITNode("/", address_cells), description)

    def addImage(self, name, description=None, hash=HASH_ALGO, arch=None):
        self.getRoot().addImage(name, description, hash=hash, arch=arch)

    def setImage(self, name, image, hash=HASH_ALGO):
        self.getRoot().setImage(name, image, hash=hash)

    def setImageEntry(self, name, address):
        self.getRoot().setImageEntry(name, address)

    def addConfiguration(self, name, description=None, hash=HASH_ALGO):
        self.getRoot().addConfiguration(name, description, hash=hash)

    def setDefaultConfiguration(self, config):
        self.getRoot().setDefaultConfiguration(config)

    def finalizeConfigs(self, hash=HASH_ALGO):
        self.getRoot().finalizeConfigs(hash=hash)

class RamdiskITS(FITImage):

    def __init__(self, address_cells=-1):
        super().__init__("U-Boot ITS for ramdisk", address_cells)

class Binman(ITS):

    def __init__(self, description=None):
        super().__init__(DtsNode("/"), description)
        self.getRoot().set(DtsNode("binman"))
        self["binman"]["sort-by-offset"] = True
        self["binman"]["allow-repack"] = True

    def setSizeCells(self, size_cells):
        self["binman"]["#size-cells"] = size_cells
        self["binman"].setSizeCells(size_cells)

    def setFilename(self, fname):
        self["binman"]["filename"] = fname

    def setIntProperty(self, name, value, base=16):
        prop = DtsIntProperty(name, value, base)
        self["binman"].set(prop)

    def parseETypeParams(self, params):
        return self.getRoot().parseParameter(params)

    def setEntry(self, entry):
        self["binman"].set(entry)

    def setEType(self, etype):
        (image, arg) = self.parseETypeParams(etype)

        # if there is a version number or a date in the
        # file name then the last image is considered
        # to be latest/newest image:
        img = glob.glob(image.replace("\\*", "*"))
        if len(img) > 0:
            image = sorted(img)[-1]

        image_size = 0

        if "add-size" in arg:
            if not os.path.isfile(image):
                print(f"Error: {image}: no such file!", file=sys.stderr)
                sys.exit(1)

            image_size = os.stat(image).st_size

        name = "binary"
        offset = 0
        size = 0
        min_size = 0
        align_size = 0
        align_end = 0
        align = 0

        if "name" in arg:
            name = arg["name"]

        if "offset" in arg:
            offset = eval(arg["offset"])

        if "size" in arg:
            size = eval(arg["size"])

        if "min-size" in arg:
            min_size = eval(arg["min-size"])

        if "add-size" in arg:
            sz = max(image_size, min_size)
            min_size = eval(arg["add-size"]) + sz

        if "align-size" in arg:
            align_size = eval(arg["align-size"])

        if "align-end" in arg:
            align_size = eval(arg["align-end"])

        if "align" in arg:
            align = eval(arg["align"])

        entry = DtsNode(name)
        entry["filename"] = image
        entry["offset"] = offset

        if size > 0:
            entry["size"] = size

        if min_size > 0:
            prop = DtsIntProperty("min-size", min_size)
            #print("min_size is", min_size)

            # Force multi-cell format for large values
            if min_size > 0xffffffff:
                #print("min_size is > 32bit")
                prop.setCells(2)  # Force 2 cells
                #print(f"Forced property to use 2 cells")

            entry.set(prop)

        if align_size > 0:
            entry["align-size"] = align_size

        if align_end > 0:
            entry["align-end"] = align_end

        if align > 0:
            entry["align"] = align

        if name == "null":
            entry["size"] = 0

        if "overlap" in arg:
            entry["overlap"] = arg["overlap"]

        self.setEntry(entry)

class MkImage(Binman):

    def __init__(self, description=None):
        super().__init__(description)
        self.mkimage = DtsNode("mkimage")
        self["binman"].set(self.mkimage)

    def parseETypeParams(self, params):
        return self.mkimage.parseParameter(params)

    def setEntry(self, entry):
        self["binman"]["mkimage"].set(entry)

class DeviceTree:

    def __init__(self, dtb_file):
        self.dtb_file = dtb_file
        self.fdt = None
        self.root = None

    def open(self):
        try:
            self.fdt = libfdt.Fdt(open(self.dtb_file, mode='rb').read())
            self.root = self.fdt.path_offset("/")
        except libfdt.FdtException as e:
            raise e

    def getFdt(self):
        if not self.fdt:
            try:
                self.open()
            except libfdt.FdtException as e:
                raise e
        return self.fdt

    def getRoot(self):
        if not self.root:
            self.root = self.getFdt().path_offset("/")
        return self.root

    def getModel(self):
        try:
            return self.getFdt().getprop(self.getRoot(), "model").as_str().strip()
        except libfdt.FdtError as e:
            return None
        except libfdt.FdtException as e:
            return None
        except AttributeError as e:
            return None
        return None

    def getCompatible(self):
        try:
            return self.getFdt().getprop(self.getRoot(), "compatible").as_stringlist()
        except libfdt.FdtError as e:
            return None
        except libfdt.FdtException as e:
            return None
        except AttributeError as e:
            return None
        return None

def usage():
    prog = os.path.basename(sys.argv[0])
    print("{0} {1}\n".format(prog, __version__))
    print("usage: {0} [options] ...".format(prog))
    print("\nOptions:")
    print("  -h|--help                  print this help")

def main():

    filename = None
    kernel = []
    fdt = []
    ramdisk = []
    etype = []
    binman = None
    mkimage = None
    args = None
    hash = HASH_ALGO
    pad_byte = 0xff
    fill_byte = -1
    align = 0
    align_size = 0
    align_end = 0
    dtc_file = None
    arch = "arm64"
    address_cells = -1
    size_cells = -1

    try:
        opts, argv = getopt.getopt(sys.argv[1:],
                                   "hf:",
                                   ["help",
                                    "no-hash",
                                    "file=",
                                    "binman=",
                                    "mkimage=",
                                    "args=",
                                    "kernel=",
                                    "fdt=",
                                    "ramdisk=",
                                    "etype=",
                                    "hash=",
                                    "fill=",
                                    "pad=",
                                    "align=",
                                    "align-size=",
                                    "align-end=",
                                    "dtc=",
                                    "arch=",
                                    "addrcells=",
                                    "size-cells="])
    except getopt.GetoptError as err:
        print("Error: {0}".format(err), file=sys.stderr)
        usage()
        sys.exit(1)

    for option, optarg in opts:
        if option in ("-h", "--help"):
            usage()
            sys.exit(0)
        elif option in ("-f", "--file"):
            filename = optarg
        elif option in ("--binman"):
            binman = optarg
        elif option in ("--mkimage"):
            mkimage = optarg
        elif option in ("--args"):
            args = optarg
        elif option in ("--kernel"):
            kernel.append(optarg)
        elif option in ("--fdt"):
            fdt.append(optarg)
        elif option in ("--ramdisk"):
            ramdisk.append(optarg)
        elif option in ("--etype"):
            etype.append(optarg)
        elif option in ("--dtc"):
            dtc_file = optarg
        elif option in ("--hash"):
            hash = optarg
        elif option in ("--no-hash"):
            hash = None
        elif option in ("--fill"):
            fill_byte = eval(optarg)
        elif option in ("--pad"):
            pad_byte = eval(optarg)
        elif option in ("--align"):
            align = eval(optarg)
        elif option in ("--align-size"):
            align_size = eval(optarg)
        elif option in ("--align-end"):
            align_end = eval(optarg)
        elif option in ("--arch"):
            arch = optarg
        elif option in ("--addrcells"):
            address_cells = eval(optarg)
        elif option in ("--size-cells"):
            size_cells = eval(optarg)
        else:
            print("Error: unknown option \"{0}\"!".format(option), file=sys.stderr)
            sys.exit(1)

    if binman or mkimage:

        if binman:
            its = Binman()
            its["binman"]["filename"] = binman
            if size_cells > 0:
                its["binman"]["#size-cells"] = size_cells
        elif mkimage:
            its = MkImage()
            its["binman"]["mkimage"]["filename"] = mkimage
            if size_cells > 0:
                its["binman"]["#size-cells"] = size_cells
            if args:
                its["binman"]["mkimage"]["args"] = args

        its.setIntProperty("pad-byte", pad_byte)

        if align > 0:
            its.setIntProperty("align", align)

        if align_size > 0:
            its.setIntProperty("align-size", align_size)

        if align_end > 0:
            its.setIntProperty("align-end", align_end)

        if fill_byte > 0:
            its.setIntProperty("fill-byte", fill_byte)

        for i in range(len(etype)):
            its.setEType(etype[i])

    else:

        its = RamdiskITS(address_cells)

        for i in range(len(kernel)):

            if ".efi" in kernel:
                its.setDescription("EFI firmware bootloader")
                name = "efi-{0}".format(i)
                its.addImage(name, "EFI firmware", hash=hash, arch=arch)
            else:
                its.setDescription("U-Boot ITS Linux Kernel Image")
                name = "kernel-{0}".format(i)
                its.addImage(name, "Linux kernel", hash=hash, arch=arch)

            its.setImage(name, kernel[i], hash=hash)

        for i in range(len(fdt)):
            name = "fdt-{0}".format(i)

            description = None

            if has_libfdt and os.path.isfile(fdt[i]):
                dtb = DeviceTree(fdt[i])
                description = dtb.getModel()

            if not description:
                description = "Flattened Device Tree blob"

            its.addImage(name, description, hash=hash, arch=arch)
            its.setImage(name, fdt[i], hash=hash)

        for i in range(len(ramdisk)):
            name = "ramdisk-{0}".format(i)
            its.addImage(name, "Ramdisk image", hash=hash, arch=arch)
            its.setImage(name, ramdisk[i], hash=hash)

        its.finalizeConfigs(hash=hash)

    if filename:
        with open(filename, "w") as fh:
            its.write(fh)

        if dtc_file:
            return subprocess.run(["dtc",
                                   "--force",
                                   "--out", dtc_file,
                                   filename]).returncode
    else:
        its.write(sys.stdout)

    return 0

if __name__ == "__main__":
    sys.exit(main())

#######################################
# Local Variables:
# mode: Python
# tab-width: 4
# End:
# kate: space-indent on; indent-width 4; mixedindent off; indent-mode python;
# vi: set tabstop=4 shiftwidth=4:
# vim: set filetype=python:
